package algoprog.base;

/**
 * Une classe pour des files statiques. Une file statique est une file dont
 * on connat  l'avance le nombre maximum d'lments ajoutables
 * conscutivement. Ce nombre s'appelle la taille de la file. Le nombre
 * d'lments contenus dans la file  un instant donn est la longueur de la
 * file. Quand la file a une longueur gale  sa taille, on dit qu'elle est
 * pleine.
 * Sauf mention contraire, toutes les mthodes sont en O(1).
 * (voir le <a href="../../../html/asd/base/FileStatique.html">source</a>).
 * @author  2002 Marc Gatano
 * @version 3.0 (26/09/02)
 */
public class FileStatique {

    private Object[] table; // le tableau contenant les lments de la file

    private int entree; // entree = l'indice dans table du prochain lment
                        // ajout
    private int sortie; // table[sortie] = le suivant de la file

    // table[sortie..entree-1] sont les lments significatifs de la file.
    // C'est une tranche d'un tableau *circulaire*.

    private static final int DEFAUT = 64;
    
    /**
     * Construit une file vide de taille maximum une valeur par dfaut.
     */
    public FileStatique() {
	table = new Object[DEFAUT];
	entree = 0;
	sortie = 0;
    }

    /**
     * Construit une file vide de taille donne.
     */
    public FileStatique(int taille) {
	table = new Object[taille+1];
	entree = 0;
	sortie = 0;
	
    }

    /**
     * Retourne la valeur (i + 1) modulo table.length
     */
    private int plusUn(int i) {
	if ( i == table.length - 1 ){
	    return 0;
	}
	else {
	    return i + 1;
	}
    }
    
    /**
     * Ajoute un nouvel objet dans la file.
     */
    public void ajouter(Object objet) {
	table[entree] = objet;
	entree = plusUn(entree);
	
    }

    /**
     * Retire un objet de la file et retourne cet objet.
     * Prcondition : la file n'est pas vide.
     */
    public Object retirer() {
        Object x = table[sortie];
	sortie = plusUn(sortie);
        return x;
    }

    /**
     * Retourne l'objet situ en tete de file.
     * Prcondition : la file n'est pas vide.
     */
    public Object suivant() {
	return table[sortie];
    }

    /**
     * Retourne le nombre d'lments contenus dans la file.
     */
    public int longueur() {
	if ( entree >= sortie ) {
	    return entree - sortie;
        }
	else {
	    return table.length - sortie + entree;
        }
    }
    
    /**
     * Retourne true si la file ne contient aucun objet, false sinon.
     */
    public boolean vide () {
	return entree == sortie;
    }

    /**
     * Retourne true si la file est pleine, false sinon.
     */
    public boolean pleine() {
	return sortie == plusUn(entree);
    }

    /**
     * Retourne une copie de la file.
     * Complexit : O(longueur()).
     */
    public Object clone() {
        FileStatique f = new FileStatique(table.length);

        f.entree = entree;
	f.sortie = sortie;

	int i = sortie;
	
	while ( i != entree ) {
	    f.table[i] = table[i];
	    i = plusUn(i);
	}

        return f;
    }

    /**
     * Retourne une reprsentation de la file sous forme de String.
     * Complexit : O(longueur()).
     */
    public String toString() {
	String s = " ->";
	int i = sortie;
	
	while ( i != entree ) {
	    s = " " + table[i] + s;
	    i = plusUn(i);
	}

	return "->" + s;
    }
}
