package fr.unice.iut.shapes.providers;

import fr.unice.iut.shapes.exception.ShapeAlreadyExistException;
import fr.unice.iut.shapes.resources.Point;
import fr.unice.iut.shapes.resources.Shape;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Classe charger de gerer la liste des formes mises a disposition par le service.
 * Ajout - Suppression - Requetage.
 */
public class ShapesProvider {

    /*
     * Map contenant l'ensemble des formes connues par le service.
     * Dans un service complet, il faudrait mettre en place un systeme de persistance (base de donnees, ...)
     */
    private static Map<String, Shape> allShapes = new HashMap<String, Shape>();

    public static List<Shape> getAllShapes() {
        return new ArrayList<Shape>(allShapes.values());
    }

    /**
     * Definit une nouvelle {@link Shape} dans le service.
     *
     * @param shape - L'objet {@link Shape} a ajouter
     * @throws ShapeAlreadyExistException si une forme avec le meme identifiant etait deja presente.
     */
    public static void createShape(Shape shape) throws ShapeAlreadyExistException {
        if (findShapeById(shape.getId()) != null) {
            throw new ShapeAlreadyExistException();
        } else {
            allShapes.put(shape.getId(), shape);
        }
    }

    /**
     * Supprime une {@link Shape} de la liste des formes mises a disposition par le service.
     *
     * @param shape - L'objet {@link Shape} a supprimer
     * @return true si la forme a bien ete supprime, false sinon (elle n'etait pas definie)
     */
    public static boolean deleteShape(Shape shape) {
        return allShapes.remove(shape.getId()) != null;
    }

    /**
     * Cherche dans les formes disponibles celle qui correspond a l'identifiant donne.
     *
     * @param id - L'identifiant de l'objet {{@link Shape}}
     * @return la forme, ou null s'il n'existe pas de forme avec cet identifiant.
     */
    public static Shape findShapeById(String id) {
        return allShapes.get(id);
    }

    // Claase utilitaire: on n'instancie pas
    private ShapesProvider() {
    }

    // Definition de quelques formes de base
    static {
        Shape shape = new Shape("square");
        shape.addPoint(new Point(0.0, 0.0));
        shape.addPoint(new Point(0.0, 1.0));
        shape.addPoint(new Point(1.0, 1.0));
        shape.addPoint(new Point(1.0, 0.0));
        allShapes.put(shape.getId(), shape);

        shape = new Shape("triangle");
        shape.addPoint(new Point(0.0, 0.0));
        shape.addPoint(new Point(2.0, 2.0));
        shape.addPoint(new Point(4.0, 0.0));
        allShapes.put(shape.getId(), shape);

        shape = new Shape("france");
        shape.addPoint(new Point(464.696678, 7.898770));
        shape.addPoint(new Point(464.611531, 7.898425));
        shape.addPoint(new Point(456.271820, 8.718423));
        shape.addPoint(new Point(434.417616, 20.948826));
        shape.addPoint(new Point(435.707378, 27.612398));
        shape.addPoint(new Point(433.376681, 35.368403));
        shape.addPoint(new Point(436.060878, 69.616315));
        shape.addPoint(new Point(436.060545, 69.635908));
        shape.addPoint(new Point(436.060418, 69.643301));
        shape.addPoint(new Point(434.497148, 81.150522));
        shape.addPoint(new Point(427.836655, 88.184025));
        shape.addPoint(new Point(421.072930, 98.868598));
        shape.addPoint(new Point(421.072787, 98.879319));
        shape.addPoint(new Point(421.072570, 98.895586));
        shape.addPoint(new Point(405.337178, 110.211029));
        shape.addPoint(new Point(383.574380, 116.664725));
        shape.addPoint(new Point(370.246507, 118.813091));
        shape.addPoint(new Point(344.268848, 134.060444));
        shape.addPoint(new Point(339.144818, 146.674391));
        shape.addPoint(new Point(339.389236, 153.011901));
        shape.addPoint(new Point(351.801498, 158.383264));
        shape.addPoint(new Point(351.793050, 158.688553));
        shape.addPoint(new Point(351.791823, 158.732905));
        shape.addPoint(new Point(337.352249, 165.232688));
        shape.addPoint(new Point(327.383944, 171.272791));
        shape.addPoint(new Point(318.477569, 172.803572));
        shape.addPoint(new Point(308.471886, 167.218881));
        shape.addPoint(new Point(277.048927, 161.777841));
        shape.addPoint(new Point(274.955540, 159.907273));
        shape.addPoint(new Point(262.681911, 163.987291));
        shape.addPoint(new Point(262.568263, 163.981994));
        shape.addPoint(new Point(262.541778, 163.980760));
        shape.addPoint(new Point(251.963764, 145.466280));
        shape.addPoint(new Point(256.813526, 137.344868));
        shape.addPoint(new Point(254.816502, 129.564852));
        shape.addPoint(new Point(241.871396, 128.727953));
        shape.addPoint(new Point(239.554602, 132.059784));
        shape.addPoint(new Point(233.259112, 133.319759));
        shape.addPoint(new Point(213.773602, 124.257634));
        shape.addPoint(new Point(218.149859, 135.053620));
        shape.addPoint(new Point(216.829089, 145.471244));
        shape.addPoint(new Point(220.651271, 158.955372));
        shape.addPoint(new Point(227.076856, 167.721951));
        shape.addPoint(new Point(230.028695, 175.377640));
        shape.addPoint(new Point(231.949640, 195.865526));
        shape.addPoint(new Point(231.155826, 208.685926));
        shape.addPoint(new Point(234.289547, 224.650382));
        shape.addPoint(new Point(234.372279, 231.718802));
        shape.addPoint(new Point(229.974444, 230.653592));
        shape.addPoint(new Point(229.938366, 230.651693));
        shape.addPoint(new Point(229.850013, 230.647040));
        shape.addPoint(new Point(221.592484, 233.495050));
        shape.addPoint(new Point(212.355350, 231.635542));
        shape.addPoint(new Point(213.420024, 223.064805));
        shape.addPoint(new Point(209.236720, 224.689169));
        shape.addPoint(new Point(202.909340, 234.115275));
        shape.addPoint(new Point(204.320157, 239.376029));
        shape.addPoint(new Point(204.317601, 239.420702));
        shape.addPoint(new Point(204.317073, 239.429931));
        shape.addPoint(new Point(204.118835, 238.528345));
        shape.addPoint(new Point(202.934554, 237.083855));
        shape.addPoint(new Point(202.435934, 236.138010));
        shape.addPoint(new Point(202.400472, 236.133720));
        shape.addPoint(new Point(202.401899, 236.108984));
        shape.addPoint(new Point(199.388538, 229.053510));
        shape.addPoint(new Point(193.329579, 229.069879));
        shape.addPoint(new Point(194.724519, 231.986424));
        shape.addPoint(new Point(194.723336, 232.006358));
        shape.addPoint(new Point(194.719745, 232.066901));
        shape.addPoint(new Point(189.331249, 234.789239));
        shape.addPoint(new Point(187.251606, 230.264226));
        shape.addPoint(new Point(182.380966, 229.179405));
        shape.addPoint(new Point(181.987529, 224.891634));
        shape.addPoint(new Point(171.736739, 226.990542));
        shape.addPoint(new Point(155.286236, 233.858795));
        shape.addPoint(new Point(150.742489, 225.124431));
        shape.addPoint(new Point(134.766325, 205.655552));
        shape.addPoint(new Point(127.691705, 206.214879));
        shape.addPoint(new Point(125.715780, 201.719254));
        shape.addPoint(new Point(112.266079, 207.636046));
        shape.addPoint(new Point(105.732951, 204.557701));
        shape.addPoint(new Point(102.111363, 217.375304));
        shape.addPoint(new Point(97.892762, 219.334642));
        shape.addPoint(new Point(97.808180, 219.327844));
        shape.addPoint(new Point(97.768800, 219.324678));
        shape.addPoint(new Point(80.408473, 216.221150));
        shape.addPoint(new Point(78.026031, 212.100594));
        shape.addPoint(new Point(62.703318, 217.647178));
        shape.addPoint(new Point(38.641174, 218.261050));
        shape.addPoint(new Point(41.663207, 222.138707));
        shape.addPoint(new Point(28.923388, 224.196086));
        shape.addPoint(new Point(25.051749, 227.318969));
        shape.addPoint(new Point(23.365631, 245.804651));
        shape.addPoint(new Point(49.623636, 241.122211));
        shape.addPoint(new Point(44.002463, 247.997686));
        shape.addPoint(new Point(50.003346, 246.754896));
        shape.addPoint(new Point(55.893201, 255.798165));
        shape.addPoint(new Point(39.931012, 251.899596));
        shape.addPoint(new Point(37.176887, 246.018983));
        shape.addPoint(new Point(33.733342, 254.194904));
        shape.addPoint(new Point(44.353239, 256.523071));
        shape.addPoint(new Point(52.127346, 261.413630));
        shape.addPoint(new Point(53.653135, 266.417151));
        shape.addPoint(new Point(51.500413, 271.444981));
        shape.addPoint(new Point(47.409726, 269.348669));
        shape.addPoint(new Point(27.424331, 271.435391));
        shape.addPoint(new Point(22.607020, 273.713454));
        shape.addPoint(new Point(33.463532, 278.869879));
        shape.addPoint(new Point(37.042125, 275.277380));
        shape.addPoint(new Point(44.308664, 286.622352));
        shape.addPoint(new Point(44.733073, 300.074418));
        shape.addPoint(new Point(56.369866, 300.428551));
        shape.addPoint(new Point(61.907687, 293.744678));
        shape.addPoint(new Point(70.312657, 296.354879));
        shape.addPoint(new Point(77.652791, 302.996802));
        shape.addPoint(new Point(81.855614, 303.289221));
        shape.addPoint(new Point(85.217430, 299.132349));
        shape.addPoint(new Point(98.172828, 307.328972));
        shape.addPoint(new Point(99.665214, 303.501831));
        shape.addPoint(new Point(99.664219, 303.514737));
        shape.addPoint(new Point(99.663140, 303.528750));
        shape.addPoint(new Point(103.731756, 313.921234));
        shape.addPoint(new Point(113.655255, 314.570872));
        shape.addPoint(new Point(117.792083, 319.849252));
        shape.addPoint(new Point(120.646641, 311.582195));
        shape.addPoint(new Point(124.143245, 314.188565));
        shape.addPoint(new Point(118.895115, 317.933127));
        shape.addPoint(new Point(123.015389, 330.144577));
        shape.addPoint(new Point(139.028024, 327.942149));
        shape.addPoint(new Point(141.389145, 324.543663));
        shape.addPoint(new Point(152.028929, 326.431910));
        shape.addPoint(new Point(148.595116, 332.380343));
        shape.addPoint(new Point(144.657668, 332.434981));
        shape.addPoint(new Point(142.589670, 337.151155));
        shape.addPoint(new Point(166.261246, 336.623795));
        shape.addPoint(new Point(166.236967, 342.800704));
        shape.addPoint(new Point(166.396063, 342.810909));
        shape.addPoint(new Point(166.400273, 342.811179));
        shape.addPoint(new Point(159.693247, 350.120673));
        shape.addPoint(new Point(165.626947, 355.118587));
        shape.addPoint(new Point(162.377310, 357.634069));
        shape.addPoint(new Point(170.623281, 359.482651));
        shape.addPoint(new Point(175.648712, 364.112852));
        shape.addPoint(new Point(184.114665, 361.528234));
        shape.addPoint(new Point(183.725495, 375.907563));
        shape.addPoint(new Point(191.569592, 379.930261));
        shape.addPoint(new Point(191.566381, 379.986365));
        shape.addPoint(new Point(191.584510, 379.987426));
        shape.addPoint(new Point(192.858066, 381.167219));
        shape.addPoint(new Point(194.128355, 382.427926));
        shape.addPoint(new Point(195.213305, 385.322147));
        shape.addPoint(new Point(195.212389, 385.338388));
        shape.addPoint(new Point(195.211785, 385.349093));
        shape.addPoint(new Point(182.961211, 398.327449));
        shape.addPoint(new Point(183.875023, 405.420733));
        shape.addPoint(new Point(192.555581, 413.805064));
        shape.addPoint(new Point(200.251412, 425.486969));
        shape.addPoint(new Point(203.267313, 435.144958));
        shape.addPoint(new Point(218.568616, 446.912726));
        shape.addPoint(new Point(222.201926, 447.128330));
        shape.addPoint(new Point(243.291112, 461.481481));
        shape.addPoint(new Point(249.333438, 456.806099));
        shape.addPoint(new Point(249.330443, 456.872930));
        shape.addPoint(new Point(249.323809, 457.020990));
        shape.addPoint(new Point(249.342404, 457.021846));
        shape.addPoint(new Point(249.341196, 457.048800));
        shape.addPoint(new Point(249.386799, 457.050901));
        shape.addPoint(new Point(249.434236, 457.053086));
        shape.addPoint(new Point(249.561878, 457.112611));
        shape.addPoint(new Point(249.654702, 457.117253));
        shape.addPoint(new Point(249.799738, 457.204581));
        shape.addPoint(new Point(249.799095, 457.218981));
        shape.addPoint(new Point(249.798533, 457.231535));
        shape.addPoint(new Point(249.907715, 457.290202));
        shape.addPoint(new Point(249.906511, 457.317156));
        shape.addPoint(new Point(250.019156, 457.295323));
        shape.addPoint(new Point(250.019898, 457.278707));
        shape.addPoint(new Point(250.020343, 457.268738));
        shape.addPoint(new Point(250.098210, 457.191287));
        shape.addPoint(new Point(250.432733, 457.206638));
        shape.addPoint(new Point(250.499923, 457.371402));
        shape.addPoint(new Point(250.498277, 457.408325));
        shape.addPoint(new Point(250.490327, 457.586664));
        shape.addPoint(new Point(250.642578, 457.512614));
        shape.addPoint(new Point(250.680973, 457.487364));
        shape.addPoint(new Point(250.683193, 457.437518));
        shape.addPoint(new Point(250.684558, 457.406872));
        shape.addPoint(new Point(250.923004, 457.067414));
        shape.addPoint(new Point(251.164479, 457.078458));
        shape.addPoint(new Point(250.820580, 457.278426));
        shape.addPoint(new Point(250.815900, 457.386617));
        shape.addPoint(new Point(250.797201, 457.385021));
        shape.addPoint(new Point(250.796806, 457.393883));
        shape.addPoint(new Point(250.792417, 457.492467));
        shape.addPoint(new Point(250.670184, 457.729579));
        shape.addPoint(new Point(250.538809, 457.750199));
        shape.addPoint(new Point(250.314096, 457.362880));
        shape.addPoint(new Point(250.314293, 457.358450));
        shape.addPoint(new Point(250.315298, 457.335927));
        shape.addPoint(new Point(250.187579, 457.276418));
        shape.addPoint(new Point(250.167796, 457.302148));
        shape.addPoint(new Point(250.167054, 457.318764));
        shape.addPoint(new Point(250.166593, 457.329102));
        shape.addPoint(new Point(249.957346, 457.427528));
        shape.addPoint(new Point(248.935628, 456.545039));
        shape.addPoint(new Point(248.908504, 456.543787));
        shape.addPoint(new Point(248.898442, 456.543323));
        shape.addPoint(new Point(248.876218, 456.623325));
        shape.addPoint(new Point(248.857760, 456.622473));
        shape.addPoint(new Point(248.851725, 456.756871));
        shape.addPoint(new Point(248.861198, 456.757308));
        shape.addPoint(new Point(248.870184, 456.757723));
        shape.addPoint(new Point(241.104736, 472.101012));
        shape.addPoint(new Point(246.024446, 472.736866));
        shape.addPoint(new Point(251.591907, 483.283651));
        shape.addPoint(new Point(250.182633, 496.985099));
        shape.addPoint(new Point(244.716522, 500.882159));
        shape.addPoint(new Point(245.597950, 507.603833));
        shape.addPoint(new Point(238.407172, 508.208708));
        shape.addPoint(new Point(243.593859, 519.120741));
        shape.addPoint(new Point(266.673005, 540.421277));
        shape.addPoint(new Point(272.174372, 553.923398));
        shape.addPoint(new Point(272.173502, 553.945549));
        shape.addPoint(new Point(272.170156, 554.030831));
        shape.addPoint(new Point(275.100225, 574.423102));
        shape.addPoint(new Point(279.823528, 584.947249));
        shape.addPoint(new Point(276.600670, 582.478707));
        shape.addPoint(new Point(269.764396, 565.968494));
        shape.addPoint(new Point(269.511414, 561.326789));
        shape.addPoint(new Point(249.371998, 535.170094));
        shape.addPoint(new Point(244.170279, 535.713299));
        shape.addPoint(new Point(241.848075, 554.594601));
        shape.addPoint(new Point(232.481411, 620.700323));
        shape.addPoint(new Point(240.365169, 607.444976));
        shape.addPoint(new Point(247.287947, 618.342027));
        shape.addPoint(new Point(237.586812, 616.875078));
        shape.addPoint(new Point(231.911680, 628.967013));
        shape.addPoint(new Point(231.828054, 635.438346));
        shape.addPoint(new Point(231.820096, 635.613983));
        shape.addPoint(new Point(231.810558, 635.613159));
        shape.addPoint(new Point(231.800838, 635.612697));
        shape.addPoint(new Point(225.427033, 670.524847));
        shape.addPoint(new Point(214.293179, 715.420741));
        shape.addPoint(new Point(208.770013, 725.398658));
        shape.addPoint(new Point(208.769656, 725.406033));
        shape.addPoint(new Point(208.766119, 725.479052));
        shape.addPoint(new Point(190.493603, 740.160250));
        shape.addPoint(new Point(191.689662, 742.805585));
        shape.addPoint(new Point(217.590976, 754.640467));
        shape.addPoint(new Point(210.228391, 768.394149));
        shape.addPoint(new Point(212.309072, 772.415742));
        shape.addPoint(new Point(221.824059, 770.650449));
        shape.addPoint(new Point(250.236589, 782.290592));
        shape.addPoint(new Point(260.816447, 782.285932));
        shape.addPoint(new Point(264.233780, 790.792135));
        shape.addPoint(new Point(271.023508, 796.383455));
        shape.addPoint(new Point(286.451693, 799.059976));
        shape.addPoint(new Point(292.690479, 795.907780));
        shape.addPoint(new Point(293.043141, 795.907861));
        shape.addPoint(new Point(294.077303, 796.236156));
        shape.addPoint(new Point(303.199285, 802.535630));
        shape.addPoint(new Point(306.217538, 807.698372));
        shape.addPoint(new Point(338.761063, 808.957159));
        shape.addPoint(new Point(347.124611, 811.210925));
        shape.addPoint(new Point(347.527229, 810.999792));
        shape.addPoint(new Point(347.743895, 811.061614));
        shape.addPoint(new Point(361.195240, 812.353843));
        shape.addPoint(new Point(359.739851, 803.256814));
        shape.addPoint(new Point(361.066365, 797.627773));
        shape.addPoint(new Point(373.331267, 799.128759));
        shape.addPoint(new Point(374.972772, 799.409081));
        shape.addPoint(new Point(375.151736, 799.844072));
        shape.addPoint(new Point(410.049175, 811.059423));
        shape.addPoint(new Point(416.435905, 822.639377));
        shape.addPoint(new Point(419.704412, 816.941329));
        shape.addPoint(new Point(437.453505, 821.025494));
        shape.addPoint(new Point(440.322424, 823.876299));
        shape.addPoint(new Point(440.733795, 824.494540));
        shape.addPoint(new Point(440.153442, 824.626933));
        shape.addPoint(new Point(436.182900, 831.626521));
        shape.addPoint(new Point(452.564578, 838.991371));
        shape.addPoint(new Point(455.343565, 845.481248));
        shape.addPoint(new Point(478.368818, 838.898258));
        shape.addPoint(new Point(490.754935, 847.458695));
        shape.addPoint(new Point(497.860570, 845.782649));
        shape.addPoint(new Point(504.555136, 846.917141));
        shape.addPoint(new Point(502.922387, 843.120182));
        shape.addPoint(new Point(511.396205, 840.403325));
        shape.addPoint(new Point(518.505849, 835.063251));
        shape.addPoint(new Point(530.726036, 834.529065));
        shape.addPoint(new Point(534.653169, 838.993672));
        shape.addPoint(new Point(530.028478, 818.225106));
        shape.addPoint(new Point(530.488810, 799.332058));
        shape.addPoint(new Point(530.486118, 799.004023));
        shape.addPoint(new Point(530.485500, 798.928748));
        shape.addPoint(new Point(530.343702, 796.108425));
        shape.addPoint(new Point(530.245878, 796.243596));
        shape.addPoint(new Point(529.988826, 798.664387));
        shape.addPoint(new Point(529.989111, 798.699442));
        shape.addPoint(new Point(529.989264, 798.718260));
        shape.addPoint(new Point(529.950330, 798.799041));
        shape.addPoint(new Point(529.870859, 798.746214));
        shape.addPoint(new Point(529.790596, 798.612208));
        shape.addPoint(new Point(529.790319, 798.577892));
        shape.addPoint(new Point(529.789946, 798.531767));
        shape.addPoint(new Point(529.608485, 798.103406));
        shape.addPoint(new Point(529.451948, 798.319507));
        shape.addPoint(new Point(529.452360, 798.370797));
        shape.addPoint(new Point(529.432589, 798.373548));
        shape.addPoint(new Point(528.949686, 802.300637));
        shape.addPoint(new Point(524.475693, 799.380915));
        shape.addPoint(new Point(526.689717, 796.568738));
        shape.addPoint(new Point(526.689547, 796.546229));
        shape.addPoint(new Point(526.689517, 796.542170));
        shape.addPoint(new Point(533.012938, 777.945298));
        shape.addPoint(new Point(529.942694, 780.875266));
        shape.addPoint(new Point(524.915343, 772.853698));
        shape.addPoint(new Point(527.069900, 768.805641));
        shape.addPoint(new Point(533.942635, 776.109433));
        shape.addPoint(new Point(544.122700, 762.620363));
        shape.addPoint(new Point(544.145827, 762.620102));
        shape.addPoint(new Point(544.201242, 762.619478));
        shape.addPoint(new Point(551.038697, 757.404090));
        shape.addPoint(new Point(563.975962, 755.667530));
        shape.addPoint(new Point(572.458783, 745.722355));
        shape.addPoint(new Point(595.532706, 730.047770));
        shape.addPoint(new Point(604.415617, 728.564169));
        shape.addPoint(new Point(604.419809, 728.564075));
        shape.addPoint(new Point(604.454719, 728.563288));
        shape.addPoint(new Point(605.600993, 733.673187));
        shape.addPoint(new Point(607.367583, 735.729690));
        shape.addPoint(new Point(613.652529, 737.544176));
        shape.addPoint(new Point(613.664755, 737.543880));
        shape.addPoint(new Point(613.730689, 737.542284));
        shape.addPoint(new Point(635.829323, 738.037831));
        shape.addPoint(new Point(640.857213, 747.036221));
        shape.addPoint(new Point(660.233947, 746.274818));
        shape.addPoint(new Point(656.491199, 744.459755));
        shape.addPoint(new Point(664.064353, 739.047570));
        shape.addPoint(new Point(671.017953, 748.492968));
        shape.addPoint(new Point(691.429879, 746.319395));
        shape.addPoint(new Point(701.091070, 758.824464));
        shape.addPoint(new Point(711.616328, 763.314356));
        shape.addPoint(new Point(716.138278, 761.238533));
        shape.addPoint(new Point(716.255333, 761.233503));
        shape.addPoint(new Point(716.257341, 761.260387));
        shape.addPoint(new Point(726.191471, 767.174033));
        shape.addPoint(new Point(730.306140, 772.584688));
        shape.addPoint(new Point(736.088608, 771.323261));
        shape.addPoint(new Point(732.779579, 767.790071));
        shape.addPoint(new Point(750.949347, 769.427203));
        shape.addPoint(new Point(758.226017, 764.918890));
        shape.addPoint(new Point(784.950375, 757.064090));
        shape.addPoint(new Point(786.112689, 753.741404));
        shape.addPoint(new Point(780.442106, 748.374805));
        shape.addPoint(new Point(789.304982, 741.391018));
        shape.addPoint(new Point(790.097150, 735.341230));
        shape.addPoint(new Point(800.900498, 732.803340));
        shape.addPoint(new Point(803.844236, 727.645970));
        shape.addPoint(new Point(803.842629, 727.617584));
        shape.addPoint(new Point(803.841210, 727.592516));
        shape.addPoint(new Point(805.517274, 721.593615));
        shape.addPoint(new Point(815.340813, 718.220615));
        shape.addPoint(new Point(817.148281, 712.397234));
        shape.addPoint(new Point(824.149587, 706.325491));
        shape.addPoint(new Point(842.736117, 696.458872));
        shape.addPoint(new Point(841.155786, 689.297793));
        shape.addPoint(new Point(851.255521, 676.465051));
        shape.addPoint(new Point(854.821730, 665.370611));
        shape.addPoint(new Point(852.454675, 657.082274));
        shape.addPoint(new Point(849.154658, 657.059692));
        shape.addPoint(new Point(837.046891, 662.738970));
        shape.addPoint(new Point(824.605446, 662.126506));
        shape.addPoint(new Point(821.257175, 659.316258));
        shape.addPoint(new Point(805.038958, 653.478507));
        shape.addPoint(new Point(804.311833, 650.068590));
        shape.addPoint(new Point(796.669623, 642.128214));
        shape.addPoint(new Point(796.710370, 642.014047));
        shape.addPoint(new Point(797.154332, 640.769527));
        shape.addPoint(new Point(799.480136, 634.713977));
        shape.addPoint(new Point(793.504999, 626.300706));
        shape.addPoint(new Point(798.377577, 621.655502));
        shape.addPoint(new Point(798.772376, 614.103834));
        shape.addPoint(new Point(798.825217, 614.056616));
        shape.addPoint(new Point(799.646260, 613.323256));
        shape.addPoint(new Point(806.884590, 609.918271));
        shape.addPoint(new Point(801.819862, 600.394708));
        shape.addPoint(new Point(802.536675, 597.440222));
        shape.addPoint(new Point(787.793725, 592.149586));
        shape.addPoint(new Point(782.069068, 579.190132));
        shape.addPoint(new Point(777.691898, 578.470146));
        shape.addPoint(new Point(774.185300, 571.362316));
        shape.addPoint(new Point(774.260393, 571.277071));
        shape.addPoint(new Point(774.773583, 570.695456));
        shape.addPoint(new Point(783.637129, 566.419936));
        shape.addPoint(new Point(795.879629, 563.611651));
        shape.addPoint(new Point(799.197257, 559.382715));
        shape.addPoint(new Point(803.354175, 559.706859));
        shape.addPoint(new Point(807.786587, 555.236549));
        shape.addPoint(new Point(810.219761, 540.745802));
        shape.addPoint(new Point(801.180359, 533.913649));
        shape.addPoint(new Point(796.633175, 528.009773));
        shape.addPoint(new Point(797.227545, 518.511068));
        shape.addPoint(new Point(783.071354, 506.158398));
        shape.addPoint(new Point(783.095197, 505.472116));
        shape.addPoint(new Point(783.200790, 502.426858));
        shape.addPoint(new Point(795.758606, 495.054002));
        shape.addPoint(new Point(795.867800, 487.021903));
        shape.addPoint(new Point(789.861754, 477.512097));
        shape.addPoint(new Point(780.281887, 470.829472));
        shape.addPoint(new Point(780.418737, 464.555294));
        shape.addPoint(new Point(783.578430, 456.009517));
        shape.addPoint(new Point(770.332957, 440.764026));
        shape.addPoint(new Point(759.790033, 441.368041));
        shape.addPoint(new Point(742.501718, 451.370259));
        shape.addPoint(new Point(745.800674, 464.619557));
        shape.addPoint(new Point(738.132028, 471.477937));
        shape.addPoint(new Point(723.643175, 475.123572));
        shape.addPoint(new Point(723.542359, 474.726892));
        shape.addPoint(new Point(723.644328, 474.629947));
        shape.addPoint(new Point(726.218867, 465.844240));
        shape.addPoint(new Point(732.654701, 459.470093));
        shape.addPoint(new Point(735.204699, 450.129717));
        shape.addPoint(new Point(731.759861, 447.539815));
        shape.addPoint(new Point(729.485617, 446.880628));
        shape.addPoint(new Point(730.358977, 445.161280));
        shape.addPoint(new Point(731.402499, 440.361819));
        shape.addPoint(new Point(732.655415, 437.320952));
        shape.addPoint(new Point(734.983812, 434.062482));
        shape.addPoint(new Point(733.769832, 432.949043));
        shape.addPoint(new Point(732.013743, 431.342171));
        shape.addPoint(new Point(753.370296, 410.995555));
        shape.addPoint(new Point(751.526763, 396.113542));
        shape.addPoint(new Point(768.180740, 381.728479));
        shape.addPoint(new Point(782.163230, 365.906498));
        shape.addPoint(new Point(783.038532, 359.482706));
        shape.addPoint(new Point(789.643598, 351.902217));
        shape.addPoint(new Point(778.655158, 353.356809));
        shape.addPoint(new Point(782.272256, 347.578413));
        shape.addPoint(new Point(782.080510, 345.260079));
        shape.addPoint(new Point(782.417776, 345.204833));
        shape.addPoint(new Point(787.243847, 338.097028));
        shape.addPoint(new Point(793.601165, 338.180580));
        shape.addPoint(new Point(794.173763, 337.780296));
        shape.addPoint(new Point(795.391917, 338.204579));
        shape.addPoint(new Point(797.253213, 343.171936));
        shape.addPoint(new Point(801.648743, 344.360523));
        shape.addPoint(new Point(823.348025, 328.789813));
        shape.addPoint(new Point(818.472721, 308.940559));
        shape.addPoint(new Point(822.126743, 287.613119));
        shape.addPoint(new Point(819.303138, 284.309049));
        shape.addPoint(new Point(819.368307, 276.212465));
        shape.addPoint(new Point(819.341415, 276.041206));
        shape.addPoint(new Point(819.937280, 275.357114));
        shape.addPoint(new Point(831.006900, 232.698180));
        shape.addPoint(new Point(840.135663, 212.836790));
        shape.addPoint(new Point(846.843740, 208.213445));
        shape.addPoint(new Point(852.545546, 193.529865));
        shape.addPoint(new Point(854.889366, 191.414357));
        shape.addPoint(new Point(846.146643, 189.723698));
        shape.addPoint(new Point(835.886052, 183.983799));
        shape.addPoint(new Point(830.868952, 186.377588));
        shape.addPoint(new Point(826.480730, 184.443422));
        shape.addPoint(new Point(819.115244, 186.197015));
        shape.addPoint(new Point(816.792422, 185.563869));
        shape.addPoint(new Point(816.163257, 183.738941));
        shape.addPoint(new Point(809.914473, 181.866262));
        shape.addPoint(new Point(806.924957, 175.134565));
        shape.addPoint(new Point(795.372194, 180.227016));
        shape.addPoint(new Point(779.869711, 182.337714));
        shape.addPoint(new Point(777.878119, 175.110848));
        shape.addPoint(new Point(770.759412, 172.166903));
        shape.addPoint(new Point(758.007936, 173.278954));
        shape.addPoint(new Point(749.034005, 163.821110));
        shape.addPoint(new Point(746.353522, 154.558079));
        shape.addPoint(new Point(741.787445, 151.565937));
        shape.addPoint(new Point(728.400756, 147.324419));
        shape.addPoint(new Point(719.230984, 149.079559));
        shape.addPoint(new Point(716.695467, 152.557127));
        shape.addPoint(new Point(710.236616, 153.299623));
        shape.addPoint(new Point(704.747317, 148.989684));
        shape.addPoint(new Point(704.374875, 149.157127));
        shape.addPoint(new Point(702.507717, 148.823944));
        shape.addPoint(new Point(700.576920, 144.980704));
        shape.addPoint(new Point(691.320153, 144.980819));
        shape.addPoint(new Point(686.332226, 148.993400));
        shape.addPoint(new Point(678.645220, 149.347691));
        shape.addPoint(new Point(677.808801, 150.301269));
        shape.addPoint(new Point(677.513757, 150.151726));
        shape.addPoint(new Point(676.463619, 143.686658));
        shape.addPoint(new Point(675.563710, 143.362739));
        shape.addPoint(new Point(673.037668, 139.416842));
        shape.addPoint(new Point(672.476609, 138.889565));
        shape.addPoint(new Point(671.375889, 138.441249));
        shape.addPoint(new Point(664.330538, 131.560621));
        shape.addPoint(new Point(655.152517, 129.835023));
        shape.addPoint(new Point(646.631755, 122.098442));
        shape.addPoint(new Point(638.141746, 122.898066));
        shape.addPoint(new Point(639.524470, 112.814944));
        shape.addPoint(new Point(635.418528, 107.636402));
        shape.addPoint(new Point(638.199608, 88.134652));
        shape.addPoint(new Point(634.922907, 86.860189));
        shape.addPoint(new Point(626.478667, 96.828119));
        shape.addPoint(new Point(627.061299, 102.265807));
        shape.addPoint(new Point(616.995710, 108.388829));
        shape.addPoint(new Point(600.030268, 107.753806));
        shape.addPoint(new Point(598.834353, 108.244496));
        shape.addPoint(new Point(596.616236, 108.622662));
        shape.addPoint(new Point(595.104940, 106.569594));
        shape.addPoint(new Point(593.874996, 106.687023));
        shape.addPoint(new Point(593.321917, 106.466770));
        shape.addPoint(new Point(593.042717, 106.355758));
        shape.addPoint(new Point(593.046278, 106.350496));
        shape.addPoint(new Point(598.197236, 96.446068));
        shape.addPoint(new Point(593.119593, 83.606105));
        shape.addPoint(new Point(597.362131, 79.535996));
        shape.addPoint(new Point(582.683604, 71.329023));
        shape.addPoint(new Point(576.479881, 72.951198));
        shape.addPoint(new Point(562.783324, 71.078220));
        shape.addPoint(new Point(563.192087, 67.190272));
        shape.addPoint(new Point(559.002816, 56.843807));
        shape.addPoint(new Point(544.718155, 57.511445));
        shape.addPoint(new Point(539.152538, 54.119590));
        shape.addPoint(new Point(537.270818, 38.222819));
        shape.addPoint(new Point(530.298770, 28.793071));
        shape.addPoint(new Point(517.060867, 35.173551));
        shape.addPoint(new Point(508.528107, 35.012007));
        shape.addPoint(new Point(504.109695, 26.842334));
        shape.addPoint(new Point(499.467125, 26.770452));
        shape.addPoint(new Point(498.545147, 13.559851));
        shape.addPoint(new Point(493.404322, 0.000000));
        shape.addPoint(new Point(464.708004, 7.898815));
        shape.addPoint(new Point(464.696678, 7.898770));
        allShapes.put(shape.getId(), shape);

        shape = new Shape("star");
        shape.addPoint(new Point(3.0, 6.0));
        shape.addPoint(new Point(4.0, 4.0));
        shape.addPoint(new Point(6.0, 4.0));
        shape.addPoint(new Point(4.5, 2.5));
        shape.addPoint(new Point(5.75, 0.0));
        shape.addPoint(new Point(3.0, 1.25));
        shape.addPoint(new Point(0.25, 0.0));
        shape.addPoint(new Point(1.5, 2.5));
        shape.addPoint(new Point(0.0, 4.0));
        shape.addPoint(new Point(2.0, 4.0));
        allShapes.put(shape.getId(), shape);
    }
}
